# Padrões e Convenções - Backend

## Nomenclatura

- **Controllers**: `*Controller.php` (ex: `ResumoController.php`)
- **Use Cases**: `*UseCase.php` (ex: `ResumoUseCase.php`)
- **Repositories**: `*Repository.php` (ex: `ResumoRepository.php`)
- **Entities**: PascalCase (ex: `DProduto.php`)
- **DTOs**: `*DTO.php` (ex: `FilterDTO.php`)

## Estrutura de Código

```php
<?php
namespace App\Application\UseCase\Pobj;

// 1. Imports (agrupados)
use App\Domain\DTO\FilterDTO;
use App\Repository\Pobj\ResumoRepository;

// 2. Classe
class ResumoUseCase
{
    // 3. Propriedades privadas
    private $resumoRepository;
    
    // 4. Construtor
    public function __construct(ResumoRepository $resumoRepository)
    {
        $this->resumoRepository = $resumoRepository;
    }
    
    // 5. Métodos públicos
    public function handle(?FilterDTO $filters = null): array
    {
        // Implementação
    }
    
    // 6. Métodos privados
    private function helperMethod(): void
    {
        // Implementação
    }
}
```

## Tratamento de Erros

**Sempre usar exceções customizadas:**

```php
// ❌ ERRADO
return $this->error('Produto não encontrado', 404);

// ✅ CORRETO
$this->throwNotFound('Produto não encontrado', 'produto');
```

## Queries SQL

**Sempre usar prepared statements:**

```php
// ❌ ERRADO (SQL Injection)
$sql = "SELECT * FROM produtos WHERE id = {$id}";

// ✅ CORRETO
$sql = "SELECT * FROM produtos WHERE id = :id";
$params = ['id' => $id];
$result = $connection->executeQuery($sql, $params);
```

## Boas Práticas

1. **Controllers finos**: Apenas recebem request e delegam
2. **Lógica em Use Cases**: Toda lógica de negócio em Use Cases
3. **Acesso a dados em Repositories**: Nunca fazer queries diretas em Controllers ou Use Cases
4. **DTOs para transferência**: Usar DTOs para transferir dados entre camadas
5. **Exceções customizadas**: Sempre usar exceções customizadas em vez de retornar erros

---

**Próximo**: [Padrões Frontend](./02-frontend.md)
