# Frontend - Types (TypeScript)

Definições de tipos TypeScript para garantir type safety em toda a aplicação.

## Localização

```
src/types/
├── index.ts           # Tipos principais (POBJ)
├── omega.ts           # Tipos do sistema Omega
├── encantabra.ts      # Tipos do Encantabra
└── html2pdf.d.ts      # Declarações de tipos para html2pdf.js
```

## Tipos Principais (index.ts)

### FilterOption

Opção de filtro genérica.

```typescript
type FilterOption = {
  id: string
  nome: string
  id_segmento?: string
  id_diretoria?: string
  id_regional?: string
  id_agencia?: string
  id_gestor?: string
  funcional?: string
  id_familia?: string
  id_indicador?: string
}
```

### InitData

Dados de inicialização do sistema.

```typescript
type InitData = {
  segmentos: any[]
  diretorias: any[]
  regionais: any[]
  agencias: any[]
  gerentes_gestao: any[]
  gerentes: any[]
  familias: any[]
  indicadores: any[]
  subindicadores: any[]
  status_indicadores: any[]
  grupos: any[]
}
```

### Produto

Representa um produto/indicador.

```typescript
type Produto = {
  id: string
  id_familia: string
  familia: string
  id_indicador: string
  indicador: string
  id_subindicador?: string
  subindicador?: string
  metrica: string
  peso: number
  meta?: number
  realizado?: number
  pontos?: number
  pontos_meta?: number
  variavel_meta?: number
  variavel_realizado?: number
  ating?: number
  atingido?: boolean
  ultima_atualizacao?: string
}
```

### ResumoPayload

Payload completo do resumo.

```typescript
type ResumoPayload = {
  cards: Produto[]
  classifiedCards: ProdutoMensal[]
  variableCard: Variavel[]
  businessSnapshot: BusinessSnapshot
}
```

### DetalhesItem

Item de detalhes de contrato.

```typescript
type DetalhesItem = {
  registro_id?: string
  id_contrato?: string
  data?: string
  competencia?: string
  segmento_id?: string
  segmento?: string
  diretoria_id?: string
  diretoria_nome?: string
  // ... outros campos
  valor_realizado?: number
  valor_meta?: number
  status_id?: number
}
```

### RankingItem

Item de ranking.

```typescript
type RankingItem = {
  data?: string
  competencia?: string
  segmento?: string
  segmento_id?: string
  diretoria_id?: string
  diretoria_nome?: string
  gerencia_id?: string
  gerencia_nome?: string
  agencia_id?: string
  agencia_nome?: string
  gerente_gestao_id?: string
  gerente_gestao_nome?: string
  gerente_id?: string
  gerente_nome?: string
  rank?: number
  pontos?: number
  realizado_mensal?: number
  meta_mensal?: number
}
```

### Period

Período de datas.

```typescript
type Period = {
  start: string  // ISO date string
  end: string   // ISO date string
}
```

### BusinessSnapshot

Snapshot do negócio (dias úteis, etc.).

```typescript
type BusinessSnapshot = {
  total: number        // Total de dias úteis
  elapsed: number     // Dias úteis decorridos
  remaining: number   // Dias úteis restantes
  monthStart: string
  monthEnd: string
  today: string
}
```

## Tipos de Filtros

### ProdutoFilters

Filtros para produtos.

```typescript
type ProdutoFilters = {
  segmento?: string
  diretoria?: string
  regional?: string
  agencia?: string
  gerenteGestao?: string
  gerente?: string
  familia?: string
  indicador?: string
  subindicador?: string
  dataInicio?: string
  dataFim?: string
  status?: string
}
```

### DetalhesFilters

Filtros para detalhes.

```typescript
type DetalhesFilters = {
  segmento?: string
  diretoria?: string
  regional?: string
  agencia?: string
  gerente?: string
  gerenteGestao?: string
  familia?: string
  indicador?: string
  subindicador?: string
  dataInicio?: string
  dataFim?: string
}
```

### RankingFilters

Filtros para ranking.

```typescript
type RankingFilters = {
  segmento?: string
  diretoria?: string
  regional?: string
  agencia?: string
  gerenteGestao?: string
  gerente?: string
  grupo?: string
  dataInicio?: string
  dataFim?: string
}
```

## Tipos Omega (omega.ts)

### OmegaUser

Usuário do sistema Omega.

```typescript
type OmegaUser = {
  id: string
  nome: string
  funcional?: string
  cargo?: string
  usuario?: boolean
  analista?: boolean
  supervisor?: boolean
  admin?: boolean
  // ... outros campos de permissão
}
```

### OmegaTicket

Chamado/ticket do Omega.

```typescript
type OmegaTicket = {
  id: string
  subject?: string
  company?: string
  status?: string
  priority?: string
  opened?: string
  updated?: string
  due_date?: string
  requester_id?: string
  owner_id?: string
  team_id?: string
  // ... outros campos
}
```

### OmegaStatus

Status de chamado.

```typescript
type OmegaStatus = {
  id: string
  label: string
  tone?: string
  descricao?: string
  ordem?: number
  departamento_id?: string
}
```

## Tipos Encantabra (encantabra.ts)

Tipos específicos do módulo Encantabra (regulamentos).

## Uso em Componentes

### Exemplo: Tipando Props

```vue
<script setup lang="ts">
import type { Produto, ProdutoFilters } from '@/types'

interface Props {
  produto: Produto
  filters?: ProdutoFilters
}

const props = defineProps<Props>()
</script>
```

### Exemplo: Tipando Composables

```typescript
import { ref } from 'vue'
import type { ResumoPayload, ProdutoFilters } from '@/types'
import { getResumo } from '@/services/resumoService'

export function useResumoData(filters?: ProdutoFilters) {
  const data = ref<ResumoPayload | null>(null)
  const loading = ref(false)
  
  const load = async () => {
    loading.value = true
    try {
      data.value = await getResumo(filters)
    } finally {
      loading.value = false
    }
  }
  
  return { data, loading, load }
}
```

### Exemplo: Tipando Services

```typescript
import type { ResumoPayload, ProdutoFilters } from '@/types'
import { apiGet } from './api'

export async function getResumo(
  filters?: ProdutoFilters
): Promise<ResumoPayload | null> {
  return apiGet<ResumoPayload>('/api/pobj/resumo', filters)
}
```

## Boas Práticas

### ✅ Sempre Use Types

```typescript
// ✅ CORRETO
import type { Produto } from '@/types'
const produto: Produto = await getProduto()

// ❌ ERRADO
const produto: any = await getProduto() // Perde type safety
```

### ✅ Use Type Imports

```typescript
// ✅ CORRETO (tree-shaking)
import type { Produto } from '@/types'

// ❌ ERRADO (importa valor desnecessariamente)
import { Produto } from '@/types'
```

### ✅ Estenda Types Quando Necessário

```typescript
// Estender tipo existente
type ProdutoComExtras = Produto & {
  customField: string
}
```

### ✅ Use Union Types

```typescript
type Status = 'ativo' | 'inativo' | 'pendente'
type ViewType = 'cards' | 'table' | 'ranking'
```

## Extensões

Para adicionar novos tipos:

1. Adicione o tipo no arquivo apropriado (`index.ts`, `omega.ts`, etc.)
2. Exporte o tipo
3. Documente o uso
4. Use em componentes/services

```typescript
// Exemplo: Novo tipo
export type NovoItem = {
  id: string
  nome: string
  valor: number
}
```

---

**Anterior**: [Utils](./06-utils.md)  
**Voltar**: [Estrutura](./01-estrutura.md)
