# Frontend - Router e Config

## Router

```typescript
import { createRouter, createWebHistory } from 'vue-router'
import type { RouteRecordRaw } from 'vue-router'

const routes: RouteRecordRaw[] = [
  {
    path: '/',
    name: 'Home',
    component: () => import('../views/Home.vue'),
    meta: {
      title: 'Resumo',
      requiresAuth: false
    }
  },
  {
    path: '/detalhes',
    name: 'Detalhes',
    component: () => import('../views/Detalhes.vue'),
    alias: ['/table'], // Alias para compatibilidade
    meta: {
      title: 'Detalhes',
      requiresAuth: false
    }
  },
  {
    path: '/ranking',
    name: 'Ranking',
    component: () => import('../views/Ranking.vue')
  },
  {
    path: '/omega',
    name: 'Omega',
    component: () => import('../views/Omega.vue'),
    meta: {
      hideGlobalChrome: true // Esconde header/footer
    }
  },
  {
    path: '/encantabra',
    name: 'Encantabra',
    component: () => import('../views/Encantabra.vue')
  },
  {
    path: '/:pathMatch(.*)*',
    name: 'NotFound',
    component: () => import('../views/NotFound.vue')
  }
]

const router = createRouter({
  history: createWebHistory(import.meta.env.BASE_URL),
  routes,
  scrollBehavior(to, from, savedPosition) {
    if (savedPosition) {
      return savedPosition
    } else {
      return { top: 0 }
    }
  }
})

export default router
```

## Configuração da API

```typescript
// src/config/api.ts

export function getApiBaseUrl(): string {
  // 1. Prioridade: window.API_HTTP_BASE (configuração global)
  if (typeof window !== 'undefined' && (window as any).API_HTTP_BASE) {
    const base = String((window as any).API_HTTP_BASE).trim()
    if (base) return base.replace(/\/$/, '')
  }
  
  // 2. Variável de ambiente
  const envUrl = import.meta.env.VITE_API_URL
  if (envUrl) return envUrl.replace(/\/$/, '')
  
  // 3. Fallback: mesma origem (dev)
  if (import.meta.env.DEV && typeof window !== 'undefined') {
    return window.location.origin.replace(/\/$/, '')
  }
  
  // 4. Fallback final
  return 'http://localhost:8081'
}

export function getApiKey(): string | null {
  // 1. window.API_KEY (configuração global)
  if (typeof window !== 'undefined' && (window as any).API_KEY) {
    const key = String((window as any).API_KEY).trim()
    if (key) return key
  }
  
  // 2. Variável de ambiente
  const envKey = import.meta.env.VITE_API_KEY
  if (envKey) return String(envKey).trim()
  
  return null
}

export const API_BASE_URL = getApiBaseUrl()
```

## Ordem de Prioridade para Configuração

1. `window.API_HTTP_BASE` / `window.API_KEY` (configuração global)
2. Variáveis de ambiente (`VITE_API_URL`, `VITE_API_KEY`)
3. `window.location.origin` (fallback para dev)
4. `http://localhost:8081` (fallback final)

---

**Voltar**: [Componentes](./04-componentes.md)  
**Próximo**: [Utils](./06-utils.md)
