# Backend - Exceptions

Todas as exceções customizadas herdam de `AppException`:

## Hierarquia de Exceções

```
AppException (base)
├── BadRequestException (400)
├── UnauthorizedException (401)
├── ForbiddenException (403)
├── NotFoundException (404)
├── ValidationException (422)
└── DatabaseException (500)
```

## Exemplo de Exceção

```php
<?php
namespace App\Exception;

class NotFoundException extends AppException
{
    public function __construct(string $message = 'Recurso não encontrado', ?string $resource = null)
    {
        parent::__construct(
            $message,
            'NOT_FOUND',
            404,
            ['resource' => $resource]
        );
    }
}

// Uso no controller:
$this->throwNotFound('Produto não encontrado', 'produto');
```

## AppException (Base)

```php
<?php
namespace App\Exception;

class AppException extends \Exception
{
    private $errorCode;
    private $details;
    private $statusCode;
    
    public function __construct(
        string $message,
        string $errorCode,
        int $statusCode = 400,
        array $details = []
    ) {
        parent::__construct($message);
        $this->errorCode = $errorCode;
        $this->statusCode = $statusCode;
        $this->details = $details;
    }
    
    public function getErrorCode(): string
    {
        return $this->errorCode;
    }
    
    public function getStatusCode(): int
    {
        return $this->statusCode;
    }
    
    public function getDetails(): array
    {
        return $this->details;
    }
    
    public function toArray(): array
    {
        return [
            'error' => $this->getMessage(),
            'code' => $this->errorCode,
            'details' => $this->details,
        ];
    }
}
```

## Uso no Controller

```php
// ❌ ERRADO
return $this->error('Produto não encontrado', 404);

// ✅ CORRETO
$this->throwNotFound('Produto não encontrado', 'produto');
```

## Exceções Disponíveis

- `BadRequestException`: Requisição inválida (400)
- `UnauthorizedException`: Não autorizado (401)
- `ForbiddenException`: Proibido (403)
- `NotFoundException`: Recurso não encontrado (404)
- `ValidationException`: Erro de validação (422)
- `DatabaseException`: Erro de banco de dados (500)

---

**Voltar**: [Backend - Estrutura](./01-estrutura.md)  
**Próximo**: [Specification Pattern](./09-specification-pattern.md)
