# Backend - Event Subscribers

## ExceptionSubscriber

Trata todas as exceções e retorna JSON padronizado:

```php
<?php
namespace App\EventSubscriber;

class ExceptionSubscriber implements EventSubscriberInterface
{
    public function onKernelException(ExceptionEvent $event): void
    {
        $exception = $event->getThrowable();
        
        // Loga exceção
        $this->logException($exception, $event->getRequest());
        
        // Cria resposta JSON
        $response = $this->createResponse($exception);
        
        $event->setResponse($response);
    }
    
    private function createResponse(\Throwable $exception): JsonResponse
    {
        // AppException customizada
        if ($exception instanceof AppException) {
            $statusCode = $exception->getStatusCode();
            $errorData = $exception->toArray();
        }
        // Exceções HTTP do Symfony
        elseif ($exception instanceof HttpExceptionInterface) {
            $statusCode = $exception->getStatusCode();
            $errorData = [
                'error' => $exception->getMessage(),
                'code' => 'HTTP_ERROR',
            ];
        }
        // Erros genéricos
        else {
            $statusCode = 500;
            $errorData = [
                'error' => $this->environment === 'dev' 
                    ? $exception->getMessage() 
                    : 'Erro interno do servidor',
                'code' => 'INTERNAL_ERROR',
            ];
        }
        
        // Adiciona debug em dev
        if ($this->environment === 'dev') {
            $errorData['debug'] = [
                'class' => get_class($exception),
                'file' => $exception->getFile(),
                'line' => $exception->getLine(),
                'trace' => $this->formatStackTrace($exception),
            ];
        }
        
        return new JsonResponse([
            'success' => false,
            'data' => $errorData
        ], $statusCode);
    }
}
```

## RateLimitSubscriber

Implementa rate limiting por IP e rota:

```php
<?php
namespace App\EventSubscriber;

class RateLimitSubscriber implements EventSubscriberInterface
{
    private $rateLimits;
    
    public function __construct(array $rateLimits)
    {
        $this->rateLimits = $rateLimits;
        // Exemplo:
        // [
        //     'default' => ['limit' => 100, 'window' => 60],
        //     'auth' => ['limit' => 5, 'window' => 60],
        //     'api' => ['limit' => 1000, 'window' => 3600],
        // ]
    }
    
    public function onKernelRequest(RequestEvent $event): void
    {
        $request = $event->getRequest();
        $path = $request->getPathInfo();
        $ip = $request->getClientIp();
        
        // Determina qual limite usar
        $config = $this->getRateLimitConfig($path);
        
        // Verifica limite (implementação com cache/Redis)
        if ($this->isRateLimitExceeded($ip, $path, $config)) {
            throw new TooManyRequestsException('Rate limit excedido');
        }
        
        // Incrementa contador
        $this->incrementRateLimit($ip, $path, $config);
    }
}
```

## CorsPreflightSubscriber

Gerencia requisições CORS preflight:

```php
<?php
namespace App\EventSubscriber;

class CorsPreflightSubscriber implements EventSubscriberInterface
{
    public function onKernelRequest(RequestEvent $event): void
    {
        $request = $event->getRequest();
        
        if ($request->getMethod() === 'OPTIONS') {
            $response = new Response();
            $response->headers->set('Access-Control-Allow-Origin', '*');
            $response->headers->set('Access-Control-Allow-Methods', 'GET, POST, PUT, DELETE, OPTIONS');
            $response->headers->set('Access-Control-Allow-Headers', 'Content-Type, X-API-Key');
            $response->setStatusCode(200);
            $event->setResponse($response);
        }
    }
}
```

## RequestIdSubscriber

Adiciona ID único a cada requisição:

```php
<?php
namespace App\EventSubscriber;

class RequestIdSubscriber implements EventSubscriberInterface
{
    public function onKernelRequest(RequestEvent $event): void
    {
        $requestId = bin2hex(random_bytes(16));
        $_SERVER['REQUEST_ID'] = $requestId;
        $event->getRequest()->headers->set('X-Request-ID', $requestId);
    }
}
```

## Event Subscribers Disponíveis

- `CorsPreflightSubscriber`: Gerencia requisições CORS preflight
- `CsrfProtectionSubscriber`: Proteção CSRF (configurável)
- `ExceptionSubscriber`: Tratamento centralizado de exceções
- `RateLimitSubscriber`: Rate limiting por IP/rota
- `RequestIdSubscriber`: Adiciona ID único às requisições

---

**Próximo**: [Exceptions](./08-exceptions.md)
