# Backend - Entities e DTOs

## Entities (Doctrine ORM)

Entidades representam tabelas do banco de dados.

### Exemplo de Entity

```php
<?php
namespace App\Entity\Pobj;

use Doctrine\ORM\Mapping as ORM;

/**
 * @ORM\Entity
 * @ORM\Table(name="d_produtos")
 */
class DProduto
{
    /**
     * @ORM\Id
     * @ORM\Column(type="integer")
     * @ORM\GeneratedValue
     */
    private $id;
    
    /**
     * @ORM\Column(type="string", length=255)
     */
    private $metrica;
    
    /**
     * @ORM\ManyToOne(targetEntity="App\Entity\Pobj\Familia")
     * @ORM\JoinColumn(name="familia_id", referencedColumnName="id")
     */
    private $familia;
    
    /**
     * @ORM\ManyToOne(targetEntity="App\Entity\Pobj\Indicador")
     * @ORM\JoinColumn(name="indicador_id", referencedColumnName="id")
     */
    private $indicador;
    
    // Getters e setters...
}
```

## FilterDTO

`FilterDTO` é a classe central para filtros. Normaliza diferentes formatos de parâmetros.

```php
<?php
namespace App\Domain\DTO;

class FilterDTO
{
    public $page;
    public $limit;
    public $filters;
    
    public function __construct(array $params = [])
    {
        // Extrai paginação
        $this->page = isset($params['page']) ? (int)$params['page'] : 1;
        $this->limit = isset($params['limit']) ? (int)$params['limit'] : 20;
        unset($params['page'], $params['limit']);
        
        $this->filters = $params;
    }
    
    // Método genérico que tenta múltiplas variações
    public function get(string $key, $default = null)
    {
        // Tenta: 'segmentoId', 'segmento', 'segmento_id'
        $keys = [
            $key . 'Id',
            $key,
            $this->normalizeKey($key),
        ];
        
        foreach ($keys as $k) {
            if (isset($this->filters[$k])) {
                return $this->filters[$k];
            }
        }
        
        return $default;
    }
    
    // Métodos específicos (type-safe)
    public function getSegmento() { return $this->get('segmento'); }
    public function getDiretoria() { return $this->get('diretoria'); }
    public function getRegional() { return $this->get('regional'); }
    public function getAgencia() { return $this->get('agencia'); }
    public function getGerente() { return $this->get('gerente'); }
    public function getDataInicio() { return $this->get('dataInicio'); }
    public function getDataFim() { return $this->get('dataFim'); }
    // ... outros
}
```

### Uso do FilterDTO

```php
$filters = new FilterDTO($request->query->all());

// Aceita qualquer uma dessas variações:
// ?segmentoId=1
// ?segmento=1
// ?segmento_id=1
$segmento = $filters->getSegmento();
```

## Outros DTOs

### CardDTO (Resumo)
```php
<?php
namespace App\Domain\DTO\Resumo;

class CardDTO
{
    public $id;
    public $indicador;
    public $meta;
    public $realizado;
    public $ating;
    public $atingido;
    
    public static function fromArray(array $data): self
    {
        $dto = new self();
        $dto->id = $data['id'] ?? null;
        $dto->indicador = $data['indicador'] ?? '';
        $dto->meta = (float)($data['meta'] ?? 0);
        $dto->realizado = (float)($data['realizado'] ?? 0);
        $dto->ating = (float)($data['ating'] ?? 0);
        $dto->atingido = (bool)($data['atingido'] ?? false);
        return $dto;
    }
    
    public function toArray(): array
    {
        return [
            'id' => $this->id,
            'indicador' => $this->indicador,
            'meta' => $this->meta,
            'realizado' => $this->realizado,
            'ating' => $this->ating,
            'atingido' => $this->atingido,
        ];
    }
}
```

## Principais Entidades

### POBJ
- `DProduto`: Produtos
- `DEstrutura`: Estrutura organizacional
- `FDetalhes`: Detalhes de indicadores
- `FRealizados`: Valores realizados
- `FMeta`: Metas
- `FPontos`: Pontos
- `FVariavel`: Variáveis

### Omega
- `OmegaUsuario`: Usuários
- `OmegaChamado`: Chamados/tickets
- `OmegaStatus`: Status dos chamados
- `OmegaDepartamento`: Departamentos

---

**Próximo**: [Security](./06-security.md)
